import os

from codeable_detectors.detected_component import DetectedComponent
from codeable_detectors.evidences import Evidences, update_keyword_args, FailedEvidence, LinkEvidence
from codeable_detectors.utils import get_all_directories, get_all_files


class Project(object):
    def __init__(self, name, root_directory):
        self.name = name
        self.root_directory = root_directory
        self.gen_model = ""
        self.all_directories = get_all_directories(root_directory)
        self.all_directories.append(os.path.realpath(root_directory))
        self.all_files = get_all_files(root_directory)
        self.all_paths = self.all_directories + self.all_files
        self.failed_evidences = Evidences()
        self.object_names = []
        self.detected_components = {}
        self.env_vars = {}

    def _process_and_generate_detected_component(self, name, detected_component, options):
        detected_component.component_types.extend(options["component_types"])
        detected_component.link_types.extend(options["link_types"])
        detected_component.technology_types.extend(options["technology_types"])

        component_types_string = "[" + ", ".join(detected_component.component_types) + "]"
        detected_component.add_name_and_compute_id_and_aliases(name, self)
        identifier = detected_component.identifier
        self.gen_model += f"{identifier} = CClass(component, '{name}', stereotypeInstances = {component_types_string})\n"
        self.object_names.append(identifier)

        self.detected_components[identifier] = detected_component
        return identifier

    def add_component(self, detectors, directory, **kwargs):
        options = update_keyword_args({'name': None, 'component_types': [], 'link_types': [],
                                     'technology_types': []}, kwargs)
        name = guessed_name = options["name"]

        if not isinstance(detectors, list):
            detectors = [detectors]
        directory = os.path.realpath(self.root_directory + "/" + directory)

        if guessed_name is None:
            guessed_name = guess_name_from_directory(directory, self.root_directory)

        detected_component = DetectedComponent(directory)

        for detector in detectors:
            evidences = detector.detect(directory, guessed_name=guessed_name)
            failed_evidence_error_prepend = "detector '" + detector.__class__.__name__ + "' failed in directory '" + directory + "': "

            if evidences.have_failed():
                self.failed_evidences.add_only_failed(evidences, failed_evidence_error_prepend)
                return None
            else:
                if len(evidences.get()) == 1:
                    evidence = evidences.get()[0]
                    detected_component.add_types(evidence)
                    detected_component.evidences.append(evidence)
                    # use detector guessed name, if none was provided or guess by previous detector
                    if name is None and evidence.name is not None:
                        name = evidence.name
                elif len(evidences.get()) > 1:
                    self.failed_evidences.add(
                        [FailedEvidence(failed_evidence_error_prepend + "multiple components found")])
                    return None
                else:
                    self.failed_evidences.add([FailedEvidence(failed_evidence_error_prepend + "no component found")])
                    return None
        if name is None:
            # no detector has set the name => use the guessed name
            name = guessed_name

        identifier = self._process_and_generate_detected_component(name, detected_component, options)
        return identifier

    def add_link(self, detectors, source, target, **kwargs):
        print("*** ADD LINK: " + str(source) + " -> " + str(target))
        if not isinstance(detectors, list):
            detectors = [detectors]
        if source is None:
            raise Exception("add_link: source cannot be 'None'")
        if target is None:
            raise Exception("add_link: target cannot be 'None'")

        directory = self.detected_components[source].directory
        evidences = Evidences()

        for detector in detectors:
            if isinstance(detector, str):
                if detector in self.detected_components:
                    # a component identifier can be provided as a detector, if the component's matches might
                    # be considered as enough evidence for a link as well (e.g., for a web server
                    # offering the component already implies possible links to the web clients)
                    evidences.add(self.detected_components[detector].get_implied_link_evidence())
                else:
                    evidences.add(FailedEvidence(
                        "detector '" + detector.__class__.__name__ + "' for link '" + source + "' to '" + target +
                        ": component '" + detector + "' not yet detected"))
            else:
                kwargs["source"] = self.detected_components[source]
                kwargs["target"] = self.detected_components[target]
                detector_evidences = detector.detect(directory, **kwargs)
                if detector_evidences.have_failed():
                    evidences.add_only_failed(detector_evidences, "detector '" + detector.__class__.__name__ +
                                              "' for link '" + source + "' to '" + target +
                                              "' failed in directory '" + directory + "': ")
                else:
                    if detector_evidences.is_empty():
                        evidences.add(FailedEvidence("detector '" + detector.__class__.__name__ +
                                                     "' for link '" + source + "' to '" + target +
                                                     "' failed in directory '" + directory + "' - none found: "))
                    elif len(detector_evidences.get()) > 1:
                        evidences.add(FailedEvidence("detector '" + detector.__class__.__name__ +
                                                     "' for link '" + source + "' to '" + target +
                                                     "' failed in directory '" + directory + "' - multiple evidences found"))
                    else:
                        if not isinstance(detector_evidences.get()[0], LinkEvidence):
                            evidences.add(FailedEvidence(
                                "evidence '" + str(detector_evidences) + "' should be a link evidence, but it is not"))
                        else:
                            evidences.add(detector_evidences.get()[0])

        if evidences.have_failed():
            self.failed_evidences.add_only_failed(evidences)
            return
        if evidences.is_empty():
            evidences.add(FailedEvidence("no evidence for link '" + source + "' to '" + target +
                                         "' found in directory '" + directory + "'"))

        link_evidence = None
        for evidence in evidences.get():
            if not link_evidence:
                link_evidence = evidence
            else:
                link_evidence.update(evidence)

        keywords_args = "roleName = \"target\""

        if link_evidence.link_types:
            keywords_args = keywords_args + ", stereotypeInstances = [" + ", ".join(link_evidence.link_types) + "]"
        if link_evidence.tagged_values:
            tagged_values_list = [(k, v) for k, v in link_evidence.tagged_values.items()]
            tagged_values_string = ", ".join([str("'" + k + "' : '" + v + "'") for k, v in tagged_values_list])
            keywords_args = keywords_args + ", taggedValues = {" + tagged_values_string + "}"

        self.gen_model += f"addLinks({{{source}: {target}}}, {keywords_args})\n"

    def add_links(self, detectors, source, targets, **kwargs):
        for target in targets:
            self.add_link(detectors, source, target, **kwargs)

    def save_as_file(self, file_name):
        source = "from codeableModels import *\n"
        source += "from metamodels.microserviceComponentsMetamodel import *\n\n"
        source += self.gen_model
        objects = "[" + ", ".join(self.object_names) + "]"
        source += f"\n\nmodel = CBundle(\"model\", elements = {objects})"
        source += "\nmodelViews = [model, {}]\n"
        print("\n\nWriting File: \n\n" + source)
        f = open(file_name, "w")
        f.write(source)
        f.close()

    def add_env_vars(self, detectors, directory):
        if not isinstance(detectors, list):
            detectors = [detectors]

        evidences = Evidences()

        directory_real_path = os.path.realpath(self.root_directory + "/" + directory)

        for detector in detectors:
            detector_evidences = detector.detect(directory_real_path)
            if detector_evidences.have_failed():
                self.failed_evidences.add_only_failed(detector_evidences, "detector '" + detector.__class__.__name__ +
                                                      "' for directory + '" + directory_real_path + "' failed: ")
                return
            evidences.add(detector_evidences)
        for match in evidences.get_all_matches():
            self.env_vars.update(match.kwargs)


def guess_name_from_directory(directory, root_directory):
    file_relative_path = os.path.relpath(directory, root_directory)
    file_name_parts = str(file_relative_path).split("/")
    if len(file_name_parts) >= 1:
        return file_name_parts[0].capitalize()
    return None
